/*
 * Copyright (C) 2003  Robert Collins  <robertc@squid-cache.org>
 * 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * DO NOT ALTER THE NEXT LINE
 * arch-tag: 35f7b455-2497-4974-999f-38dd60aa3cc6
 *
 */

#include "ConfigCVSSource.h"
#include <iostream>
#include <stdexcept>
#include <getopt++/GetOption.h>
#include <getopt++/StringOption.h>
#include <getopt++/BoolOption.h>
#include <cassert>
#include <fstream>
#include "Path.h"

using namespace std;

// pserver://anoncvs@sources.redhat.com/cvs/cygwin-apps/libgetopt++?tag=XXX
// -> -d :pserver:anoncvs@sources.redhat.com:/cvs/cygwin-apps co -r XXX libgetopt++
// scheme :pserver:
// repository anoncvs@sources.redhat.com:/cvs/cygwin-apps
// module libgetopt++
// tag XXX

void
ConfigCVSSource::processRepository()
{
    /* currently discarded */
    string::size_type optionPos = repository.rfind("?");

    if (optionPos != string::npos) {
        repository = repository.substr(0, optionPos);
    }

    string::size_type modulePos = repository.rfind("/");

    if (modulePos == string::npos)
        throw new runtime_error ("No CVS module specified");

    module = repository.substr(modulePos + 1);

    if (module.size() == 0)
        throw new runtime_error ("No CVS module specified");

    repository = repository.substr(0, modulePos);

}

ConfigCVSSource::ConfigCVSSource() 
{
}

ConfigCVSSource::ConfigCVSSource(string const &aUrl) : theUrl(aUrl)
{
    if (theUrl.find("pserver://") == 0) {
        scheme = ":pserver:";
        repository = theUrl.substr(10);
	urlPart = repository;
        processRepository();
    } else if (theUrl.find("ext://") == 0) {
        scheme = ":ext:";
        repository = theUrl.substr(6);
	urlPart = repository;
        processRepository();
    }

    if (Verbose()) {
        cout << " processed CVS url: " << scheme << " " << repository << " " << tag
        << " " << module << endl;
    }
}

string
ConfigCVSSource::url(bool const &stripScheme) const
{
    return (stripScheme ? "" : urlScheme(scheme)) + urlPart;
}

void
ConfigCVSSource::get
    (string const &where) const throw(exception *)
{
    // TODO: add support for tags
    string command = "cd `dirname " + where + "` && cvs -d " + scheme + repository + " co -d `basename " + where + "` " + module;

    run (command, "CVS");
}

string
ConfigCVSSource::urlScheme(string const &scheme) 
{
    if (scheme == ":ext:")
	return "ext://";
    return "pserver://";
}

void
ConfigCVSSource::update (string const &where) const throw(exception *)
{
    // TODO: add support for tags
    string command = "cd " + where + " && cvs -q -d " + scheme + repository + " up -Pd ";

    run (command, "CVS");
}

int
ConfigCVSSource::changes (string const &where) const throw(exception *)
{
    // logic should be... 
    // get tree version by inspection
    // use diff -r unless tag == tree version tag.
    // in which case, use diff.
    // for now, just diff.
    string command = "cd " + where + " && cvs -q -d " + scheme + repository + " diff -up";

    run (command, "CVS");
    return 2; // TODO: synthesis a result code
}

int
ConfigCVSSource::missing (string const &where) const throw(exception *)
{
    // TODO: add support for tags
    // logic should be... 
    // get tree version by inspection
    // use rdiff -r -r unless tag == tree version tag.
    // in which case, use diff -r tag.
    string command = "cd " + where + " && cvs -q -d " + scheme + repository + " diff -up ";
    if (tag.size() == 0)
	command += " -rHEAD";
    else
	command += "-r" + tag;

    run (command, "CVS");
    return 2; // TODO: synthesis a result code
}

string
ConfigCVSSource::getOneLineFile(string const & where) const
{
    ifstream buffer(where.c_str());
    string result;
    if (buffer.is_open())
	buffer >> result;
    return result;
}

string 
ConfigCVSSource::getRoot(string const &where) const
{
    return getOneLineFile(where + "/CVS/Root");
}

string 
ConfigCVSSource::getRepository(string const &where) const
{
    return getOneLineFile(where + "/CVS/Repository");
}

string
ConfigCVSSource::treeVersion(string const &where) const
{
  if (Verbose())
        cout << "getting treeVersion for CVS repo at " << where << endl;

    string result = getRoot(where);
    
    if (!result.size())
	throw new runtime_error("Could not get Root for CVS at '" + where + "'");

    result += "/";

    string repository = getRepository(where);

    if (!result.size())
	throw new runtime_error("Could not get Repository for CVS at '" + where + "'");

    result += repository;

    return schemeFromRepository(result);
}

string
ConfigCVSSource::schemeFromRepository(string &result)
{
    /* Now :pserver:->pserver:// (and ext) */
    if (result.substr(0,9) == ":pserver:")
        return "pserver://" + result.substr(9);

    if (result.substr(0,6) == ":ext:")
        return "ext://" + result.substr(6);

    return "ext://" + result;
}

void
ConfigCVSSource::ignore (ConfigSource const *, string const &where) const throw (std::exception *)
{
}

/* Repository (CVS/Root) + module + tag must remain the same */
ConfigCVSSource*
ConfigCVSSource::Create(Path const &location, ConfigSource const *mayBeParent) throw (std::exception *)
{
    ConfigCVSSource const *parent = dynamic_cast<ConfigCVSSource const *>(mayBeParent);

    ConfigCVSSource *result = new ConfigCVSSource;
    string Root = result->getRoot(location.fullName());
    if (!Root.size()) {
	delete result;
	return NULL;
    }
    string Repository = result->getRepository(location.fullName());
    if (!Repository.size()) {
	delete result;
	return NULL;
    }
    /* No parent, valid CVS dir */
    if (!parent) {
	ConfigCVSSource *realResult = new ConfigCVSSource(result->treeVersion(location.fullName()));
	delete result;
	return realResult;
    }
    string canonical = Root + "/" + Repository;
    canonical = schemeFromRepository(canonical);
    /* check for a change in parent */
    if ((canonical.find(urlScheme(parent->scheme) + parent->repository + "/" + parent->module) != 0)  ||
	/* XXX: fixme tags */
	(parent->tag != result->tag))
      {
	/* new source */
	ConfigCVSSource *realResult = new ConfigCVSSource(result->treeVersion(location.fullName()));
	delete result;
	return realResult;
      }
    delete result;
    return const_cast<ConfigCVSSource *>(parent);
}
